<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Exports\SuperAdmin\EventsExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\Events\CreateRequest;
use App\Http\Requests\ImportRequest;
use App\Imports\SuperAdmin\EventsImport;
use App\Models\Event;
use App\Models\EventCategory;
use App\Models\Therapist;
use App\Models\Tag;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;

class TherapistEventsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        $this->middleware('permission:therapist.add_event');
        $this->middleware('permission:therapist.add_event', ['only' => ['store']]);
        $this->middleware('permission:therapist.add_event', ['only' => ['update']]);
        $this->middleware('permission:therapist.add_event', ['only' => ['destroy']]);
        $this->middleware('permission:therapist.add_event', ['only' => ['export']]);
        $this->middleware('permission:therapist.add_event', ['only' => ['import']]);
    }
    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null, $therapist)
    {
        if ($req != null) {
            $therapist_events =  $therapist->therapist_events();
            if ($req->trash && $req->trash == 'with') {
                $therapist_events =  $therapist_events->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapist_events =  $therapist_events->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapist_events = $therapist_events->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapist_events = $therapist_events->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort_field != null && $req->sort_type != null) {
                $therapist_events = $therapist_events->OrderBy($req->sort_field, $req->sort_type);
            } else {
                $therapist_events = $therapist_events->OrderBy('id', 'desc');
            }
            if ($export != null) { // for export do not paginate
                $therapist_events = $therapist_events->get();
                return $therapist_events;
            }
            $therapist_events = $therapist_events->get();
            return $therapist_events;
        }
        $therapist_events = $therapist->therapist_events()->withAll()->orderBy('id', 'desc')->get();
        return $therapist_events;
    }


    /*********View All Events  ***********/
    public function index(Request $request, Therapist $therapist)
    {
        $therapist_events = $this->getter($request, null, $therapist);
        return view('super_admins.therapists.therapist_events.index', compact('therapist_events', 'therapist'));
    }

    /*********View Create Form of Event  ***********/
    public function create(Therapist $therapist)
    {
        $tags = Tag::active()->get();
        $event_categories = EventCategory::active()->get();
        return view('super_admins.therapists.therapist_events.create', compact('therapist', 'tags', 'event_categories'));
    }

    /*********Store Event  ***********/
    public function store(CreateRequest $request, Therapist $therapist)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $request->merge(['created_by_user_id' => auth()->user()->id]);
            $data = $request->all();
            $data['image'] = uploadCroppedFile($request, 'image', 'therapist_events');
            $therapist_event = $therapist->therapist_events()->create($data);
            $therapist_event->slug = Str::slug($therapist_event->name . ' ' . $therapist_event->id, '-');
            $therapist_event->save();
            foreach ($request->sponsers as $sponser) {
                $image_url = uploadArrayFile($sponser, 'image', 'event_sponsers');
                $therapist_event->sponsers()->create([
                    'name' => $sponser['name'],
                    'image' => $image_url
                ]);
            }
            $therapist_event->tags()->sync($request->tag_ids);
            $therapist_event = $therapist->therapist_events()->withAll()->find($therapist_event->id);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_events.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_events.index', $therapist->id)->with('message', 'Event Created Successfully')->with('message_type', 'success');
    }

    /*********View Event  ***********/
    public function show(Therapist $therapist, Event $therapist_event)
    {
        if ($therapist->id != $therapist_event->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapists.therapist_events.show', compact('therapist_event', 'therapist'));
    }

    /*********View Edit Form of Event  ***********/
    public function edit(Therapist $therapist, Event $therapist_event)
    {
        if ($therapist->id != $therapist_event->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        $tags = Tag::active()->get();
        $event_categories = EventCategory::active()->get();
        return view('super_admins.therapists.therapist_events.edit', compact('therapist_event', 'therapist', 'tags', 'event_categories'));
    }

    /*********Update Event  ***********/
    public function update(CreateRequest $request, Therapist $therapist, Event $therapist_event)
    {
        if ($therapist->id != $therapist_event->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $request->merge(['last_updated_by_user_id' => auth()->user()->id]);
            $data = $request->all();
            if ($request->image) {
                $data['image'] = uploadCroppedFile($request, 'image', 'therapist_events', $therapist_event->image);
            } else {
                $data['image'] = $therapist_event->image;
            }
            $therapist_event->sponsers()->delete();
            foreach ($request->sponsers as $sponser) {
                if (isset($sponser['image'])) {
                    if (is_string($sponser['image']) && $sponser['image'] != 'undefined') {
                        $image_url = $sponser['previous_image'];
                    } else {
                        $image_url = uploadArrayFile($sponser, 'image', 'event_sponsers');
                    }
                }
                $therapist_event->sponsers()->create([
                    'name' => $sponser['name'],
                    'image' => $image_url
                ]);
            }
            $therapist_event->update($data);
            $therapist_event = Event::find($therapist_event->id);
            $slug = Str::slug($therapist_event->name . ' ' . $therapist_event->id, '-');
            $therapist_event->update([
                'slug' => $slug
            ]);
            $therapist_event->tags()->sync($request->tag_ids);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_events.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_events.index', $therapist->id)->with('message', 'Event Updated Successfully')->with('message_type', 'success');
    }

    /********* Export  CSV And Excel  **********/
    public function export(Request $request)
    {
        $therapist_events = $this->getter($request, "export");
        if (in_array($request->export, ['csv,xlsx'])) {
            $extension = $request->export;
        } else {
            $extension = 'xlsx';
        }
        $filename = "therapist_events." . $extension;
        return Excel::download(new EventsExport($therapist_events), $filename);
    }
    /********* Import CSV And Excel  **********/
    public function import(ImportRequest $request)
    {
        $file = $request->file('file');
        Excel::import(new EventsImport, $file);
        return redirect()->back()->with('message', 'Event Categories imported Successfully')->with('message_type', 'success');
    }


    /*********Soft DELETE Event ***********/
    public function destroy(Therapist $therapist, Event $therapist_event)
    {
        if ($therapist->id != $therapist_event->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        $therapist_event->delete();
        return redirect()->back()->with('message', 'Event Deleted Successfully')->with('message_type', 'success');
    }

    /*********Permanently DELETE Event ***********/
    public function destroyPermanently(Request $request, Therapist $therapist, $therapist_event)
    {
        if ($therapist->id != $therapist_event->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        $therapist_event = Event::withTrashed()->find($therapist_event);
        if ($therapist_event) {
            if ($therapist_event->trashed()) {
                if ($therapist_event->image && file_exists(public_path($therapist_event->image))) {
                    unlink(public_path($therapist_event->image));
                }
                $therapist_event->forceDelete();
                return redirect()->back()->with('message', 'Event Deleted Successfully')->with('message_type', 'success');
            } else {
                return redirect()->back()->with('message', 'Event is Not in Trash')->with('message_type', 'error');
            }
        } else {
            return redirect()->back()->with('message', 'Event Not Found')->with('message_type', 'error');
        }
    }
    /********* Restore Event***********/
    public function restore(Request $request, Therapist $therapist, $therapist_event)
    {
        if ($therapist->id != $therapist_event->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        $therapist_event = Event::withTrashed()->find($therapist_event);
        if ($therapist_event->trashed()) {
            $therapist_event->restore();
            return redirect()->back()->with('message', 'Event Restored Successfully')->with('message_type', 'success');
        } else {
            return redirect()->back()->with('message', 'Event Not Found')->with('message_type', 'error');
        }
    }
}
